############################################################################
# NAME :            buex2012.ps1
#
# AUTHOR : Ben Kaufman ben.kaufman@norquest.ca
#
# DESCRIPTION :     ATO uses this to call Backup Exec 2012 jobs
#
# PARAMETERS : variable name - optional|mandatory - description
#           jid - mandatory - BackupExec 2012 job name
#           hold - optional - accepts "true" (default) or "false", this is to hold the job in backup exec after the job has run to prevent BE's scheduler from running it again
#
#           Remote Connection Parameters (all of these must be specified to use remote powershell connections)
#           svr - optional -  FQDN of BackupExec 2012 server
#           unam - optional - Username for remote powershell connection
#           pwd - optional - Password for remote powershell connection
#       
# OUTPUTS :
#           Exit Code 0 for failure, Code 1 for success
#
# PROCESS :
#                   [1]  Launch remote PS Session (if used)
#                   [2]  Call Backup Exec Job and wait
#                   [3]  Hold job (if required)
#                   [4]  Clean up remote PS Session (if used)
#                   [5]  output logs and exit
#
# EXAMPLE : 
#            powershell.exe buex2012.ps1 -jid ATO-MyJOB -svr mybackupserver.domain -unam domain\myuser -pwd mypassword -hold false
#
# NOTES :           Paramaters are not position sensitive
#                   
# CHANGES :
#     DATE         WHO              DETAIL
#     13June2012   Ben Kaufman      Original Code
#    
###############

Param($svr=, $unam='', $pwd='', $jid='', $hold="true")

$remotecon = $true

#checks to see if the jobid paramater is present or not, this is required to run this script.
if (!$jid ) {
 write-output "Job ID Parameter Missing"
 exit 0
}

#checks to see if the 3 variables required for remote communication are there, if not it will execute the be job locally
if (!$svr -or !$unam -or !$pwd ) {
 $remotecon = $false
}
write-output $("Using a Remote PS Session: " + $remotecon)

#create credential object and powershell session
if ($remotecon) {
    $SecurePassword=ConvertTo-SecureString String $pwd AsPlainText force
    $cred=New-object System.Management.Automation.PsCredential( $unam, $SecurePassword)

    $session = New-PSSession -ComputerName $svr -credential $cred -Name SessionXyz

    #try opening a remote pssession, if fails the script will exit
    Try {
        Enter-PSSession -Session $session -erroraction stop
    } catch {
    write-output "Can't Connect to remote server"
    exit 0
    }
}

Import-Module BEMCLI

$Job = Start-BEJob -InputObject $jid -confirm:$false

$output = Wait-BEJob -inputobject $job

$log = Get-BEJobHistory $job -FromLastJobRun | Get-BEJobLog

if ($hold -eq "true") {
$null = Suspend-BEJob -InputObject $jid
write-output "Holding Job"
}

if ($remotecon) {
    exit-pssession
    Remove-PSSession $session
}

if($output.JobStatus -eq "Succeeded") {
 #JOB SUCCESS here
 write-output $("JOB ID: {" + $output.JobID + "}")
 write-output $("JOB NAME: " + $output.JobName)
 write-output $log
 exit 1
 
} else {
 #JOB FAILED Here
 write-output $log
 exit 0
}
